import Sidebar from '@/src/components/layout/Sidebar';
import { Calendar, Clock, User, Globe, Briefcase, Edit, Trash2, CheckCircle, AlertCircle } from 'lucide-react';

interface Props {
    params: Promise<{ id: string }>;
}

export default async function TaskDetailsPage({ params }: Props) {
    const { id } = await params;

    // Fetch task data
    const taskResponse = await fetch(`${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/api/tasks/${id}`, {
        cache: 'no-store'
    });

    if (!taskResponse.ok) {
        return (
            <div className="min-h-screen bg-gray-50 flex">
                <Sidebar />
                <div className="flex-1 p-6" dir="rtl">
                    <div className="text-center">
                        <h1 className="text-2xl font-bold text-red-600">שגיאה</h1>
                        <p className="text-gray-600 mt-2">המשימה לא נמצאה</p>
                    </div>
                </div>
            </div>
        );
    }

    const task = await taskResponse.json();

    const formatDate = (dateString: string) => {
        const date = new Date(dateString);
        return date.toLocaleDateString('he-IL');
    };

    const getPriorityColor = (priority: string) => {
        const colors: Record<string, string> = {
            urgent: 'bg-red-100 text-red-800',
            high: 'bg-orange-100 text-orange-800',
            medium: 'bg-yellow-100 text-yellow-800',
            low: 'bg-green-100 text-green-800'
        };
        return colors[priority] || 'bg-gray-100 text-gray-800';
    };

    const getPriorityLabel = (priority: string) => {
        const labels: Record<string, string> = {
            urgent: 'דחוף',
            high: 'גבוה',
            medium: 'בינוני',
            low: 'נמוך'
        };
        return labels[priority] || priority;
    };

    const getStatusColor = (status: string) => {
        const colors: Record<string, string> = {
            pending: 'bg-gray-100 text-gray-800',
            in_progress: 'bg-blue-100 text-blue-800',
            completed: 'bg-green-100 text-green-800',
            on_hold: 'bg-yellow-100 text-yellow-800',
            cancelled: 'bg-red-100 text-red-800'
        };
        return colors[status] || 'bg-gray-100 text-gray-800';
    };

    const getStatusLabel = (status: string) => {
        const labels: Record<string, string> = {
            pending: 'ממתין',
            in_progress: 'בתהליך',
            completed: 'הושלם',
            on_hold: 'בהמתנה',
            cancelled: 'בוטל'
        };
        return labels[status] || status;
    };

    const getCategoryIcon = (category: string) => {
        const icons: Record<string, string> = {
            customer: '👤',
            domain: '🌐',
            project: '📁',
            technical: '⚙️',
            administrative: '📋',
            development: '💻',
            design: '🎨',
            other: '📌'
        };
        return icons[category] || '📌';
    };

    const getCategoryLabel = (category: string) => {
        const labels: Record<string, string> = {
            customer: 'לקוחות',
            domain: 'דומיינים',
            project: 'פרויקטים',
            technical: 'טכני',
            administrative: 'מנהלי',
            development: 'פיתוח',
            design: 'עיצוב',
            other: 'אחר'
        };
        return labels[category] || category;
    };

    return (
        <div className="min-h-screen bg-gray-50 flex">
            <Sidebar />
            <div className="flex-1 p-6" dir="rtl">
                <div className="flex items-center justify-between mb-6">
                    <div>
                        <h1 className="text-2xl font-bold">{task.title}</h1>
                        <p className="text-gray-600 mt-1">פרטי המשימה</p>
                    </div>
                    <div className="flex items-center gap-3">
                        <a
                            href={`/tasks/${id}/edit`}
                            className="inline-flex items-center gap-2 px-4 py-2 bg-yellow-600 text-white hover:bg-yellow-700 rounded-md transition-colors"
                        >
                            <Edit className="w-4 h-4" />
                            ערוך
                        </a>
                        <a
                            href={`/tasks/${id}/delete`}
                            className="inline-flex items-center gap-2 px-4 py-2 bg-red-600 text-white hover:bg-red-700 rounded-md transition-colors"
                        >
                            <Trash2 className="w-4 h-4" />
                            מחק
                        </a>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Task Details */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Basic Info */}
                        <div className="bg-white rounded-lg shadow p-6">
                            <div className="flex items-center gap-3 mb-4">
                                <span className="text-3xl">
                                    {getCategoryIcon(task.category)}
                                </span>
                                <div>
                                    <h2 className="text-xl font-semibold">{task.title}</h2>
                                    <div className="flex items-center gap-2 mt-1">
                                        <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs ${getStatusColor(task.status)}`}>
                                            {getStatusLabel(task.status)}
                                        </span>
                                        <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs ${getPriorityColor(task.priority)}`}>
                                            {getPriorityLabel(task.priority)}
                                        </span>
                                    </div>
                                </div>
                            </div>

                            {task.description && (
                                <p className="text-gray-700 mb-4">{task.description}</p>
                            )}

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                                <div className="flex items-center gap-3">
                                    <Calendar className="w-5 h-5 text-gray-400" />
                                    <div>
                                        <div className="font-medium">תאריך יצירה</div>
                                        <div className="text-gray-600">{formatDate(task.created_date)}</div>
                                    </div>
                                </div>
                                {task.due_date && (
                                    <div className="flex items-center gap-3">
                                        <Clock className="w-5 h-5 text-gray-400" />
                                        <div>
                                            <div className="font-medium">תאריך יעד</div>
                                            <div className="text-gray-600">{formatDate(task.due_date)}</div>
                                        </div>
                                    </div>
                                )}
                                <div className="flex items-center gap-3">
                                    <span className="text-lg">📂</span>
                                    <div>
                                        <div className="font-medium">קטגוריה</div>
                                        <div className="text-gray-600">{getCategoryLabel(task.category)}</div>
                                    </div>
                                </div>
                                {task.progress !== undefined && (
                                    <div className="flex items-center gap-3">
                                        <CheckCircle className="w-5 h-5 text-gray-400" />
                                        <div>
                                            <div className="font-medium">התקדמות</div>
                                            <div className="text-gray-600">{task.progress}%</div>
                                        </div>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* Progress Bar */}
                        {task.progress !== undefined && (
                            <div className="bg-white rounded-lg shadow p-6">
                                <h3 className="text-lg font-semibold mb-4">התקדמות המשימה</h3>
                                <div className="w-full bg-gray-200 rounded-full h-2.5">
                                    <div
                                        className="bg-blue-600 h-2.5 rounded-full transition-all duration-300"
                                        style={{ width: `${task.progress}%` }}
                                    ></div>
                                </div>
                                <div className="text-sm text-gray-600 mt-2">{task.progress}% הושלם</div>
                            </div>
                        )}

                        {/* Comments */}
                        {task.comments && task.comments.length > 0 && (
                            <div className="bg-white rounded-lg shadow p-6">
                                <h3 className="text-lg font-semibold mb-4">הערות</h3>
                                <div className="space-y-3">
                                    {task.comments.map((comment: any, index: number) => (
                                        <div key={index} className="p-3 bg-gray-50 rounded-lg">
                                            <div className="text-sm text-gray-600 mb-1">
                                                {formatDate(comment.created_date)}
                                            </div>
                                            <div>{comment.content}</div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        )}
                    </div>

                    {/* Sidebar */}
                    <div className="space-y-6">
                        {/* Customer Info */}
                        {task.customer_name && (
                            <div className="bg-white rounded-lg shadow p-6">
                                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                                    <User className="w-5 h-5" />
                                    לקוח
                                </h3>
                                <div className="space-y-2">
                                    <div className="font-medium">{task.customer_name}</div>
                                    {task.customer_company && (
                                        <div className="text-sm text-gray-600">{task.customer_company}</div>
                                    )}
                                </div>
                            </div>
                        )}

                        {/* Domain Info */}
                        {task.domain_name && (
                            <div className="bg-white rounded-lg shadow p-6">
                                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                                    <Globe className="w-5 h-5" />
                                    דומיין
                                </h3>
                                <div className="font-medium">{task.domain_name}</div>
                            </div>
                        )}

                        {/* Project Info */}
                        {task.project_title && (
                            <div className="bg-white rounded-lg shadow p-6">
                                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                                    <Briefcase className="w-5 h-5" />
                                    פרויקט
                                </h3>
                                <div className="font-medium">{task.project_title}</div>
                            </div>
                        )}

                        {/* Event Info */}
                        {task.event_title && (
                            <div className="bg-white rounded-lg shadow p-6">
                                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                                    <Calendar className="w-5 h-5" />
                                    אירוע קשור
                                </h3>
                                <div className="space-y-2">
                                    <div className="font-medium">{task.event_title}</div>
                                    <a
                                        href={`/events/${task.event_id}`}
                                        className="text-blue-600 hover:text-blue-800 text-sm"
                                    >
                                        צפה באירוע
                                    </a>
                                </div>
                            </div>
                        )}

                        {/* Task Stats */}
                        <div className="bg-white rounded-lg shadow p-6">
                            <h3 className="text-lg font-semibold mb-4">סטטיסטיקות</h3>
                            <div className="space-y-3 text-sm">
                                <div className="flex justify-between">
                                    <span>נוצר:</span>
                                    <span>{formatDate(task.created_date)}</span>
                                </div>
                                {task.updated_at && (
                                    <div className="flex justify-between">
                                        <span>עודכן:</span>
                                        <span>{formatDate(task.updated_at)}</span>
                                    </div>
                                )}
                                {task.due_date && (
                                    <div className="flex justify-between">
                                        <span>תאריך יעד:</span>
                                        <span>{formatDate(task.due_date)}</span>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}
